<?php

namespace App\Controller;

use App\Entity\Charge;
use App\Entity\Voyage;
use App\Form\ChargeForm;
use App\Repository\ChargeRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/charge')]
final class ChargeController extends AbstractController
{
    #[Route('/{id}',name: 'app_charge_index', methods: ['GET'])]
    public function index(ChargeRepository $chargeRepository,
                          Voyage                $voyage
): Response
    {
        $canView = $this->isGranted('CHARGE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('CHARGE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('CHARGE_UPDATE');
        $canDelete = $this->isGranted('CHARGE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        return $this->render('charge/index.html.twig', [
            'charges' => $chargeRepository->findBy(array('voyage'=>$voyage)),
            'voyage'=>$voyage
        ]);
    }

    #[Route('/{id}/new', name: 'app_charge_new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $entityManager,
                        Voyage                $voyage): Response
    {
        $canCreate = $this->isGranted('CHARGE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canCreate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }


        $today = new \DateTime();  // Récupère la date actuelle
        $today->format('Y-m-d');

        $charge = new Charge();
        $form = $this->createForm(ChargeForm::class, $charge);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $charge->setVoyage($voyage);
            $charge->setUser($this->getUser());
            $charge->setDateCharge($today);
            $entityManager->persist($charge);
            $entityManager->flush();

            return $this->redirectToRoute('app_charge_index', [
                'id'=>$voyage->getId()
            ], Response::HTTP_SEE_OTHER);
        }

        return $this->render('charge/new.html.twig', [
            'charge' => $charge,
            'form' => $form,
            'voyage'=>$voyage
        ]);
    }

    #[Route('/{id}', name: 'app_charge_show', methods: ['GET'])]
    public function show(Charge $charge): Response
    {

        $canView = $this->isGranted('CHARGE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('CHARGE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('CHARGE_UPDATE');
        $canDelete = $this->isGranted('CHARGE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        return $this->render('charge/show.html.twig', [
            'charge' => $charge,
        ]);
    }

    #[Route('/{id}/edit', name: 'app_charge_edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Charge $charge, EntityManagerInterface $entityManager): Response
    {


        $canUpdate = $this->isGranted('CHARGE_UPDATE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canUpdate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }        $form = $this->createForm(ChargeForm::class, $charge);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->flush();

            return $this->redirectToRoute('app_charge_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('charge/edit.html.twig', [
            'charge' => $charge,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'app_charge_delete', methods: ['POST'])]
    public function delete(Request $request, Charge $charge, EntityManagerInterface $entityManager): Response
    {


        $canDelete = $this->isGranted('CHARGE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        if ($this->isCsrfTokenValid('delete'.$charge->getId(), $request->getPayload()->getString('_token'))) {
            $entityManager->remove($charge);
            $entityManager->flush();
        }

        return $this->redirectToRoute('app_charge_index', [], Response::HTTP_SEE_OTHER);
    }
}
