<?php

namespace App\Controller;

use App\Entity\Chauffeur;
use App\Entity\UserSociete;
use App\Form\ChauffeurType;
use App\Repository\ChauffeurRepository;
use App\Repository\UserSocieteRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/chauffeur')]
class ChauffeurController extends AbstractController
{
    #[Route('/', name: 'app_chauffeur_index', methods: ['GET'])]
    public function index(ChauffeurRepository   $chauffeurRepository,
                          UserSocieteRepository $userSocieteRepository): Response
    {

        $canView = $this->isGranted('CHAUFFEUR_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('CHAUFFEUR_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('CHAUFFEUR_UPDATE');
        $canDelete = $this->isGranted('CHAUFFEUR_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');

        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));

        return $this->render('chauffeur/index.html.twig', [
            'chauffeurs' => $chauffeurRepository->findBy(array('societe' => $societe->getSociete())),
        ]);
    }

    #[Route('/new', name: 'app_chauffeur_new', methods: ['GET', 'POST'])]
    public function new(Request                     $request,
                        EntityManagerInterface      $entityManager,
                        UserSocieteRepository       $userSocieteRepository,
                        UserPasswordHasherInterface $userPasswordHasher): Response
    {

        $canCreate = $this->isGranted('CHAUFFEUR_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $superAmin = $this->isGranted('SUPER_ADMIN');

        if (!$canCreate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));

        $chauffeur = new Chauffeur();
        $userSociete = new UserSociete();
        $form = $this->createForm(ChauffeurType::class, $chauffeur);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $chauffeur->setSociete($societe->getSociete());
            $chauffeur->setStatut(1);

            $user = $chauffeur->getUser();

            $chauffeur->getUser()->setPassword(
                $userPasswordHasher->hashPassword(
                    $user,
                    $form->get('user')->get('plainPassword')->getData()
                )
            );
            $chauffeur->getUser()->setStatut(1);
            $chauffeur->getUser()->setRoles(['ROLE_CHAUFFEUR']);

            $userSociete->setNom($chauffeur->getNom());
            $userSociete->setPrenom($chauffeur->getPrenom());
            $userSociete->setTelephone($chauffeur->getTelephone());
            $userSociete->setStatut(1);
            $userSociete->setSociete($societe->getSociete());
            $userSociete->setUser($chauffeur->getUser());

            $entityManager->persist($chauffeur);
            $entityManager->persist($userSociete);

            $entityManager->flush();

            return $this->redirectToRoute('app_chauffeur_index', [], Response::HTTP_SEE_OTHER);
        }
        $response = new Response(null, $form->isSubmitted() ? 422 : 200);


        return $this->render('chauffeur/new.html.twig', [
            'chauffeur' => $chauffeur,
            'form' => $form,
        ], $response);
    }

    #[Route('/{id}', name: 'app_chauffeur_show', methods: ['GET'])]
    public function show(Chauffeur $chauffeur): Response
    {


        $canView = $this->isGranted('CHAUFFEUR_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('CHAUFFEUR_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('CHAUFFEUR_UPDATE');
        $canDelete = $this->isGranted('CHAUFFEUR_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        return $this->render('chauffeur/show.html.twig', [
            'chauffeur' => $chauffeur,
        ]);
    }

    #[Route('/{id}/edit', name: 'app_chauffeur_edit', methods: ['GET', 'POST'])]
    public function edit(Request                $request,
                         Chauffeur              $chauffeur,
                         EntityManagerInterface $entityManager,
                         UserSocieteRepository  $userSocieteRepository): Response
    {


        $canUpdate = $this->isGranted('CHAUFFEUR_UPDATE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canUpdate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $form = $this->createForm(ChauffeurType::class, $chauffeur);
        $form->remove('user');
        $form->handleRequest($request);


        if ($form->isSubmitted() && $form->isValid()) {

            $user = $chauffeur->getUser();
            $userSociete = $userSocieteRepository->findOneBy(array('user' => $user));
            $userSociete->setNom($chauffeur->getNom());
            $userSociete->setPrenom($chauffeur->getPrenom());
            $userSociete->setTelephone($chauffeur->getTelephone());

            $entityManager->flush();

            return $this->redirectToRoute('app_chauffeur_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('chauffeur/edit.html.twig', [
            'chauffeur' => $chauffeur,
            'form' => $form,
        ]);
    }


    #[Route('/{id}/edit/edit', name: 'app_chauffeur_edit_delete', methods: ['GET', 'POST'])]
    public function editDelete(Request                $request,
                               Chauffeur              $chauffeur,
                               EntityManagerInterface $entityManager,
                               UserSocieteRepository  $userSocieteRepository): Response
    {


        $canUpdate = $this->isGranted('CHAUFFEUR_UPDATE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canUpdate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }


        $user = $chauffeur->getUser();
        $userSociete = $userSocieteRepository->findOneBy(array('user' => $user));

        $chauffeur->setStatut(false);
        $userSociete->setStatut(false);
        $entityManager->flush();

        return $this->redirectToRoute('app_chauffeur_index', [], Response::HTTP_SEE_OTHER);

    }

    #[Route('/{id}', name: 'app_chauffeur_delete', methods: ['POST'])]
    public function delete(Request $request, Chauffeur $chauffeur, EntityManagerInterface $entityManager): Response
    {


        $canDelete = $this->isGranted('CHAUFFEUR_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        if ($this->isCsrfTokenValid('delete' . $chauffeur->getId(), $request->getPayload()->get('_token'))) {
            $entityManager->remove($chauffeur);
            $entityManager->flush();
        }

        return $this->redirectToRoute('app_chauffeur_index', [], Response::HTTP_SEE_OTHER);
    }
}
