<?php

namespace App\Controller;

use App\Entity\Pays;
use App\Entity\Province;
use App\Form\ProvinceType;
use App\Repository\ProvinceRepository;
use App\Repository\UserSocieteRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/province')]
class ProvinceController extends AbstractController
{
    #[Route('/', name: 'app_province_index', methods: ['GET'])]
    public function index(ProvinceRepository $provinceRepository,
                          UserSocieteRepository  $userSocieteRepository): Response
    {
        $canView = $this->isGranted('PROVINCE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('PROVINCE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('PROVINCE_UPDATE');
        $canDelete = $this->isGranted('PROVINCE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));

        return $this->render('province/index.html.twig', [
            'provinces' => $provinceRepository->findBy(array('societe'=>$societe->getSociete(), 'statut'=>1)),
        ]);
    }

    #[Route('/new', name: 'app_province_new', methods: ['GET', 'POST'])]
    public function new(Request                $request,
                        EntityManagerInterface $entityManager,
                        UserSocieteRepository  $userSocieteRepository): Response
    {

        $canCreate = $this->isGranted('PROVINCE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canCreate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));
        $province = new Province();
        $form = $this->createForm(ProvinceType::class, $province);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $province->setSociete($societe->getSociete());
            $province->setStatut(1);
            $entityManager->persist($province);
            $entityManager->flush();

            return $this->redirectToRoute('app_province_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('province/new.html.twig', [
            'province' => $province,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'app_province_show', methods: ['GET'])]
    public function show(Province $province): Response
    {
        $canView = $this->isGranted('PROVINCE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('PROVINCE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('PROVINCE_UPDATE');
        $canDelete = $this->isGranted('PROVINCE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        return $this->render('province/show.html.twig', [
            'province' => $province,
        ]);
    }

    #[Route('/{id}/edit', name: 'app_province_edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Province $province,
                         EntityManagerInterface $entityManager): Response
    {

        $canUpdate = $this->isGranted('PROVINCE_UPDATE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canUpdate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $form = $this->createForm(ProvinceType::class, $province);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $entityManager->flush();

            return $this->redirectToRoute('app_province_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('province/edit.html.twig', [
            'province' => $province,
            'form' => $form,
        ]);
    }


    #[Route('/{id}/province/province', name: 'app_province_edit_delete', methods: ['GET', 'POST'])]
    public function editDelete(Request $request, Province $province, EntityManagerInterface $entityManager): Response
    {


        $canDelete = $this->isGranted('PROVINCE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $province->setStatut(false);
        $entityManager->flush();

        return $this->redirectToRoute('app_province_index', [], Response::HTTP_SEE_OTHER);

    }


    #[Route('/{id}', name: 'app_province_delete', methods: ['POST'])]
    public function delete(Request $request, Province $province, EntityManagerInterface $entityManager): Response
    {


        $canDelete = $this->isGranted('PROVINCE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        if ($this->isCsrfTokenValid('delete' . $province->getId(), $request->getPayload()->get('_token'))) {
            $entityManager->remove($province);
            $entityManager->flush();
        }

        return $this->redirectToRoute('app_province_index', [], Response::HTTP_SEE_OTHER);
    }
}
