<?php

namespace App\Controller;

use App\Entity\Charge;
use App\Entity\Client;
use App\Entity\Reservation;
use App\Entity\Transit;
use App\Entity\Voyage;
use App\Form\ClientPaiementType;
use App\Form\ClientType;
use App\Form\ReservationType;
use App\Repository\ChargeRepository;
use App\Repository\ClientRepository;
use App\Repository\ReservationRepository;
use App\Repository\TransitRepository;
use App\Repository\UserSocieteRepository;
use App\Repository\VoyageRepository;
use Doctrine\ORM\EntityManagerInterface;
use Endroid\QrCode\Color\Color;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel;
use Endroid\QrCode\QrCode;
use Endroid\QrCode\RoundBlockSizeMode;
use Endroid\QrCode\Writer\PngWriter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Routing\Attribute\Route;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;



#[Route('/reservation')]
class ReservationController extends AbstractController
{

    private $entityManager;

    public function __construct(EntityManagerInterface $entityManager)
    {
        $this->entityManager = $entityManager;
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => \App\Entity\Client::class,
            'transit' => null,
        ]);

        $resolver->setRequired('transit');
        $resolver->setAllowedTypes('transit', [Transit::class]); // ✅ entité Transit
    }

    #[Route('/', name: 'app_reservation_index', methods: ['GET', 'POST'])]
    public function index(Request               $request,
                          ReservationRepository $reservationRepository,
                          UserSocieteRepository $userSocieteRepository): Response
    {
        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $reservation = new Reservation();
        $form = $this->createForm(ReservationType::class, $reservation);
        $form->remove('paied');
        $form->remove('dateReservation');
        $form->remove('statut');
        $form->remove('billnumber');
        $form->remove('transit');
        $form->remove('client');
        $form->remove('societe');

        $form->handleRequest($request);
        $reservations = $reservationRepository->findBy(array('societe' => $societe->getSociete(), 'statut' => 1));


        if ($form->isSubmitted() && $form->isValid()) {

            $dateDebut = $request->request->get("datedebut");
            $dateFin = $request->request->get("datefin");

            if (empty($dateDebut) || empty($dateFin)) {
                return $this->redirectToRoute('app_reservation_index', [], Response::HTTP_SEE_OTHER);
            }


            return $this->redirectToRoute('app_reservation_index_search', [
                'dateDebut' => $dateDebut,
                'dateFin' => $dateFin,
                // 'voyages' => $voyages,
                //  'form' => $form,

            ], Response::HTTP_SEE_OTHER);


        }


        $totalTicketsVendusPrix = $this->entityManager->createQueryBuilder()
            ->select('SUM(t.prix) as totalPrix')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->getQuery()
            ->getSingleScalarResult();

        $totalTicketsVendus = $this->entityManager->createQueryBuilder()
            ->select('COUNT(t.id) as total')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->getQuery()
            ->getSingleScalarResult();


        return $this->render('reservation/index.html.twig', [
            'reservations' => $reservations,
            'totalTicketsVendusPrix' => $totalTicketsVendusPrix,
            'totalTicketsVendus' => $totalTicketsVendus,
            'form' => $form,


        ]);
    }


    #[Route('/info/paiement/info/paiement/{id}/search/', name: 'app_reservation_infopaiement_client_search', methods: ['GET', 'POST'])]
    public function indexInfoPaiementClientSearch(Request                $request,
                                                  Transit                $transit,
                                                  ClientRepository       $clientRepository,
                                                  ReservationRepository  $reservationRepository,
                                                  TransitRepository      $transitRepository,
                                                  UserSocieteRepository  $userSocieteRepository,
                                                  EntityManagerInterface $entityManager): Response
    {
        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $clients = new Client();
        $forms = $this->createForm(ClientPaiementType::class, $clients);
        $forms->remove('code');

        $forms->handleRequest($request);


        $today = new \DateTime();
        $dateString = $today->format('dmY');

        if ($forms->isSubmitted() && $forms->isValid()) {


            $code = $request->request->get("code");
            $client = $clientRepository->findOneBy(array('code' => $code));


            return $this->redirectToRoute('app_reservation_infopaiement_client', [

                'client' => $client->getId(),
                'transit' => $transit,
                'id' => $transit->getId()


            ], Response::HTTP_SEE_OTHER);
        }


        return $this->render('reservation/search.html.twig', [
            'forms' => $forms,
            'transit' => $transit
        ]);
    }


    #[Route('/info/paiement/info/paiement/{id}/{client}/', name: 'app_reservation_infopaiement_client', methods: ['GET', 'POST'])]
    public function indexInfoPaiementClient(Request                $request,
                                            Transit                $transit,
                                            Client                 $client,
                                            ClientRepository       $clientRepository,
                                            ReservationRepository  $reservationRepository,
                                            TransitRepository      $transitRepository,
                                            UserSocieteRepository  $userSocieteRepository,
                                            EntityManagerInterface $entityManager): Response
    {
        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }










        // --- Transit (etape 1) pour ce voyage
       /* $transit = $transitRepository->findOneBy([
            'voyage' => $voyage->getId(),
            'numeroetape' => 1,
        ]);*/

        if (!$transit) {
            throw $this->createNotFoundException("Transit (étape 1) introuvable pour ce voyage.");
        }

        // --- Optionnel: si plus de place
        if ($transit->getVoyage()->getAvailable() !== null && $transit->getVoyage()->getAvailable() <= 0) {
            $this->addFlash('error', "Plus de sièges disponibles pour ce voyage.");
        }

        // --- Entities
//        $client = new Client();
        $reservation = new Reservation();
        $reservation->setTransit($transit);

        // --- Form (Client + dropdown siege disponible)
        $form = $this->createForm(ClientType::class, $client, [
            'transit' => $transit, // IMPORTANT
        ]);
        $form->remove('nom');
        $form->remove('prenom');
        $form->remove('telephone');
        $form->remove('email');

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            /** @var VehiculeSiege|null $selectedSeat */
            $selectedSeat = $form->get('vehiculesiege')->getData();
            if (!$selectedSeat) {
                $this->addFlash('error', 'Veuillez choisir un siège.');
                return $this->render('reservation/infoPaiement.html.twig', [
                    'voyage' => $transit->getVoyage()->getId(),
                    'transit' => $transit,
                    'form' => $form->createView(),
                    'reservation' => $reservation,
                ]);
            }

            // --- Dates
            $today = new \DateTimeImmutable();
            $dateString = $today->format('dmY');

            // --- Préparer client
          /*  $client->setSociete($transit->getVoyage()->getSociete());
            $client->setStatut(true); // si ton champ est bool (sinon remets 1)*/

            // --- Préparer reservation
            $reservation->setClient($client);
            $reservation->setSociete($transit->getVoyage()->getSociete());
            $reservation->setDateReservation(\DateTime::createFromImmutable($today)); // ton champ est DATE_MUTABLE
            $reservation->setStatut(true); // idem bool
            $reservation->setPaied(true);
            $reservation->setUser($this->getUser());
            $reservation->setVehiculesiege($selectedSeat);

            // --- Décrémenter available (garde-fou)
            if ($transit->getVoyage()->getAvailable() !== null) {
                if ($transit->getVoyage()->getAvailable() <= 0) {
                    $this->addFlash('error', "Plus de sièges disponibles.");
                    return $this->redirectToRoute('app_reservation_info_paiement', ['id' => $transit->getVoyage()->getId()]);
                }
                $transit->getVoyage()->setAvailable($transit->getVoyage()->getAvailable() - 1);
            }

            // --- Persist (1 seul flush idéalement)
            try {
                $entityManager->persist($client);
                $entityManager->persist($reservation);
                $entityManager->persist($transit->getVoyage());
                $entityManager->flush(); // client id généré ici
            } catch (UniqueConstraintViolationException $e) {
                // Si tu as mis la contrainte UNIQUE(transit_id, vehiculesiege_id)
                $this->addFlash('error', "Ce siège vient d’être réservé. Choisis un autre siège.");
                return $this->redirectToRoute('app_reservation_info_paiement', ['id' => $transit->getVoyage()->getId()]);
            }

            // --- Post-flush : code client + billnumber (car besoin ID)
            $client->setCode('TD' . $client->getId());

            $billnumber = $transit->getVoyage()->getSociete()->getId()
                . $client->getId()
                . $dateString;

            $reservation->setBillnumber($billnumber);

            $entityManager->flush();

            return $this->redirectToRoute('app_reservation_success_paiement', [
                'transit' => $transit->getId(),
                'reservation' => $reservation->getId(),
            ], Response::HTTP_SEE_OTHER);
        }





















/*
        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $today = new \DateTime();
        $dateString = $today->format('dmY');


        $reservation = new Reservation();
        $form = $this->createForm(ReservationType::class, $reservation);
        $form->remove('paied');
        $form->remove('dateReservation');
        $form->remove('statut');
        $form->remove('billnumber');
        $form->remove('transit');
        $form->remove('client');
        $form->remove('societe');

        $form->handleRequest($request);

        $code = $request->request->get("code");

        if ($form->isSubmitted() && $form->isValid()) {


            $billnumber = $transit->getVoyage()->getSociete()->getId() . '' . $client->getId() . '' . $dateString;

            $disponibilite = $transit->getVoyage()->getAvailable();
            $available = $disponibilite - 1;

            $voyage = $transit->getVoyage();


            $reservation->setClient($client);
            $reservation->setTransit($transit);
            $reservation->setSociete($transit->getVoyage()->getSociete());
            $reservation->setDateReservation($today);
            $reservation->setStatut(1);
            $reservation->setBillnumber($billnumber);
            $reservation->setPaied(1);

            $entityManager->persist($reservation);

            $voyage->setAvailable($available);
            $entityManager->persist($voyage);

            $entityManager->flush();


            return $this->redirectToRoute('app_reservation_success_paiement', [
                'transit' => $transit->getId(),
                'reservation' => $reservation->getId()


            ], Response::HTTP_SEE_OTHER);
        }*/


        return $this->render('reservation/infoPaiementClient.html.twig', [
            'detail' => $transit->getVoyage(),
            'transit' => $transit,
            'form' => $form,
            'reservation' => $reservation,
            'client' => $client,

        ]);
    }


    #[Route('/{id}/client/1/index/reservation', name: 'app_reservation_index_client', methods: ['GET', 'POST'])]
    public function indexClient(Request               $request,
                                Client                $client,
                                ReservationRepository $reservationRepository,
                                UserSocieteRepository $userSocieteRepository): Response
    {

        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $reservation = new Reservation();
        $form = $this->createForm(ReservationType::class, $reservation);
        $form->remove('paied');
        $form->remove('dateReservation');
        $form->remove('statut');
        $form->remove('billnumber');
        $form->remove('transit');
        $form->remove('client');
        $form->remove('societe');

        $form->handleRequest($request);
        $reservations = $reservationRepository->findBy(array('societe' => $societe->getSociete(), 'statut' => 1, 'client' => $client->getId()));


        if ($form->isSubmitted() && $form->isValid()) {

            $dateDebut = $request->request->get("datedebut");
            $dateFin = $request->request->get("datefin");

            if (empty($dateDebut) || empty($dateFin)) {
                return $this->redirectToRoute('app_reservation_index_client', [
                    'id' => $client->getId()
                ], Response::HTTP_SEE_OTHER);
            }


            return $this->redirectToRoute('app_reservation_index_search_client', [
                'dateDebut' => $dateDebut,
                'dateFin' => $dateFin,
                'id' => $client->getId()
                // 'voyages' => $voyages,
                //  'form' => $form,

            ], Response::HTTP_SEE_OTHER);


        }


        $totalTicketsVendusPrix = $this->entityManager->createQueryBuilder()
            ->select('SUM(t.prix) as totalPrix')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->andWhere('r.client = :val')
            ->setParameter('val', $client->getId())
            ->getQuery()
            ->getSingleScalarResult();

        $totalTicketsVendus = $this->entityManager->createQueryBuilder()
            ->select('COUNT(t.id) as total')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->andWhere('r.client = :val')
            ->setParameter('val', $client->getId())
            ->getQuery()
            ->getSingleScalarResult();


        return $this->render('reservation/indexClient.html.twig', [
            'reservations' => $reservations,
            'totalTicketsVendusPrix' => $totalTicketsVendusPrix,
            'totalTicketsVendus' => $totalTicketsVendus,
            'form' => $form,
            'client' => $client,


        ]);
    }


    #[Route('/{dateDebut}/{dateFin}/index/search', name: 'app_reservation_index_search', methods: ['GET', 'POST'])]
    public function indexSearch(Request               $request,
                                ReservationRepository $reservationRepository,
                                UserSocieteRepository $userSocieteRepository,
                                                      $dateDebut,
                                                      $dateFin): Response
    {

        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $reservation = new Reservation();
        $form = $this->createForm(ReservationType::class, $reservation);
        $form->remove('paied');
        $form->remove('dateReservation');
        $form->remove('statut');
        $form->remove('billnumber');
        $form->remove('transit');
        $form->remove('client');
        $form->remove('societe');

        $form->handleRequest($request);

//        $reservations = $reservationRepository->findBy(array('societe' => $societe->getSociete(), 'statut' => 1));

        $reservations = $this->entityManager->getRepository(Reservation::class)
            ->createQueryBuilder('r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
            ->andWhere('r.societe=:societe')
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('societe', $societe->getSociete())
            ->getQuery()
            ->getResult();


        $totalTicketsVendusPrix = $this->entityManager->createQueryBuilder()
            ->select('SUM(t.prix) as totalPrix')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
            ->andWhere('r.societe=:societe')
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('societe', $societe->getSociete())
            ->getQuery()
            ->getSingleScalarResult();

        $totalTicketsVendus = $this->entityManager->createQueryBuilder()
            ->select('COUNT(t.id) as total')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
            ->andWhere('r.societe=:societe')
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('societe', $societe->getSociete())
            ->getQuery()
            ->getSingleScalarResult();


        if ($form->isSubmitted() && $form->isValid()) {

            $dateDebut = $request->request->get("datedebut");
            $dateFin = $request->request->get("datefin");

            if (empty($dateDebut) || empty($dateFin)) {
                return $this->redirectToRoute('app_reservation_index', [], Response::HTTP_SEE_OTHER);
            }


            return $this->redirectToRoute('app_reservation_index_search', [
                'dateDebut' => $dateDebut,
                'dateFin' => $dateFin,
                // 'voyages' => $voyages,
                //  'form' => $form,

            ], Response::HTTP_SEE_OTHER);


        }


        return $this->render('reservation/index.html.twig', [
            'reservations' => $reservations,
            'totalTicketsVendusPrix' => $totalTicketsVendusPrix,
            'totalTicketsVendus' => $totalTicketsVendus,
            'form' => $form,


        ]);
    }


    #[Route('/{id}/{dateDebut}/{dateFin}/index/search/1/1', name: 'app_reservation_index_search_client', methods: ['GET', 'POST'])]
    public function indexSearchClient(Request               $request,
                                      Client                $client,
                                      ReservationRepository $reservationRepository,
                                      UserSocieteRepository $userSocieteRepository,
                                                            $dateDebut,
                                                            $dateFin): Response
    {

        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $reservation = new Reservation();
        $form = $this->createForm(ReservationType::class, $reservation);
        $form->remove('paied');
        $form->remove('dateReservation');
        $form->remove('statut');
        $form->remove('billnumber');
        $form->remove('transit');
        $form->remove('client');
        $form->remove('societe');

        $form->handleRequest($request);

//        $reservations = $reservationRepository->findBy(array('societe' => $societe->getSociete(), 'statut' => 1));

        $reservations = $this->entityManager->getRepository(Reservation::class)
            ->createQueryBuilder('r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
            ->andWhere('r.societe=:societe')
            ->andWhere('r.client = :val')
            ->setParameter('val', $client->getId())
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('societe', $societe->getSociete())
            ->getQuery()
            ->getResult();


        $totalTicketsVendusPrix = $this->entityManager->createQueryBuilder()
            ->select('SUM(t.prix) as totalPrix')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
            ->andWhere('r.societe=:societe')
            ->andWhere('r.client = :val')
            ->setParameter('val', $client->getId())
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('societe', $societe->getSociete())
            ->getQuery()
            ->getSingleScalarResult();

        $totalTicketsVendus = $this->entityManager->createQueryBuilder()
            ->select('COUNT(t.id) as total')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
            ->andWhere('r.societe=:societe')
            ->andWhere('r.client = :val')
            ->setParameter('val', $client->getId())
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('societe', $societe->getSociete())
            ->getQuery()
            ->getSingleScalarResult();


        if ($form->isSubmitted() && $form->isValid()) {

            $dateDebut = $request->request->get("datedebut");
            $dateFin = $request->request->get("datefin");

            if (empty($dateDebut) || empty($dateFin)) {
                return $this->redirectToRoute('app_reservation_index_client', [
                    'id' => $client->getId(),
                ], Response::HTTP_SEE_OTHER);
            }


            return $this->redirectToRoute('app_reservation_index_search_client', [
                'dateDebut' => $dateDebut,
                'dateFin' => $dateFin,
                'id' => $client->getId(),
                // 'voyages' => $voyages,
                //  'form' => $form,

            ], Response::HTTP_SEE_OTHER);


        }


        return $this->render('reservation/indexClient.html.twig', [
            'reservations' => $reservations,
            'totalTicketsVendusPrix' => $totalTicketsVendusPrix,
            'totalTicketsVendus' => $totalTicketsVendus,
            'form' => $form,
            'client' => $client


        ]);
    }


    //Brouillard de caisse


    #[Route('/brouillards/de/{id}/voyage/brouillards', name: 'app_reservation_voyage_index_brouillard', methods: ['GET', 'POST'])]
    public function indexVoyageBrouillard(Request               $request,
                                          ReservationRepository $reservationRepository,
                                          UserSocieteRepository $userSocieteRepository,
                                          ChargeRepository      $chargeRepository, Voyage $voyage): Response
    {

        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }


        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $reservation = new Reservation();
        $form = $this->createForm(ReservationType::class, $reservation);
        $form->remove('paied');
        $form->remove('dateReservation');
        $form->remove('statut');
        $form->remove('billnumber');
        $form->remove('transit');
        $form->remove('client');
        $form->remove('societe');

        $form->handleRequest($request);

        $today = new \DateTime();  // Récupère la date actuelle
        $today->format('Y-m-d');


        $reservations = $reservationRepository->findByVoyageAdmin($voyage->getId());
        $charges = $chargeRepository->findBy(array('voyage' => $voyage->getId()));

        /*
                if ($this->isGranted('SUPER_ADMIN')) {

                    $reservations = $reservationRepository->findByVoyageAdmin($voyage->getId());
                } else {
                    $reservations = $reservationRepository->findByVoyage($voyage->getId(),$this->getUser());

                }*/


        if ($form->isSubmitted() && $form->isValid()) {

            $dateDebut = $request->request->get("datedebut");
            $dateFin = $request->request->get("datefin");

            if (empty($dateDebut) || empty($dateFin)) {
                return $this->redirectToRoute('app_reservation_voyage_index', [
                    'id' => $voyage->getId(),
                ], Response::HTTP_SEE_OTHER);
            }


            return $this->redirectToRoute('app_reservation_voyage_index_search', [
                'dateDebut' => $dateDebut,
                'dateFin' => $dateFin,
                'id' => $voyage->getId(),
                // 'voyages' => $voyages,
                //  'form' => $form,

            ], Response::HTTP_SEE_OTHER);


        }


        $totalChargeVoyage = $this->entityManager->createQueryBuilder()
            ->select('SUM(c.montant) as totalMontant')
            ->from(Charge::class, 'c')
//            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
//            ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
            ->andWhere('c.voyage = :val')
//            ->andWhere('c.dateCharge >= :today')
//            ->setParameter('today', $today->format('Y-m-d'))
            ->setParameter('val', $voyage->getId())
            ->getQuery()
            ->getSingleScalarResult();

        $totalTicketsVendusPrix = $this->entityManager->createQueryBuilder()
            ->select('SUM(t.prix) as totalPrix')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
            ->andWhere('v.id = :val')
//            ->andWhere('r.dateReservation = :today')
//            ->setParameter('today', $today->format('Y-m-d'))
            ->setParameter('val', $voyage->getId())
            ->getQuery()
            ->getSingleScalarResult();

        $totalTicketsVendus = $this->entityManager->createQueryBuilder()
            ->select('COUNT(t.id) as total')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
            ->andWhere('v.id = :val')
//            ->andWhere('r.dateReservation = :today')
//            ->setParameter('today', $today->format('Y-m-d'))
            ->setParameter('val', $voyage->getId())
            ->getQuery()
            ->getSingleScalarResult();

        return $this->render('reservation/indexVoyageBrouillard.html.twig', [
            'reservations' => $reservations,
            'totalTicketsVendusPrix' => $totalTicketsVendusPrix,
            'totalTicketsVendus' => $totalTicketsVendus,
            'form' => $form,
            'voyage' => $voyage,
            'totalmontant' => $totalChargeVoyage,
            'charges' => $charges


        ]);
    }


    #[Route('/{id}/voyage', name: 'app_reservation_voyage_index', methods: ['GET', 'POST'])]
    public function indexVoyage(Request               $request,
                                ReservationRepository $reservationRepository,
                                UserSocieteRepository $userSocieteRepository,
                                Voyage                $voyage): Response
    {

        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }


        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $reservation = new Reservation();
        $form = $this->createForm(ReservationType::class, $reservation);
        $form->remove('paied');
        $form->remove('dateReservation');
        $form->remove('statut');
        $form->remove('billnumber');
        $form->remove('transit');
        $form->remove('client');
        $form->remove('societe');

        $form->handleRequest($request);

        $today = new \DateTime();  // Récupère la date actuelle
        $today->format('Y-m-d');


        $reservations = $reservationRepository->findByVoyageAdmin($voyage->getId());

        /*
                if ($this->isGranted('SUPER_ADMIN')) {

                    $reservations = $reservationRepository->findByVoyageAdmin($voyage->getId());
                } else {
                    $reservations = $reservationRepository->findByVoyage($voyage->getId(),$this->getUser());

                }*/


        if ($form->isSubmitted() && $form->isValid()) {

            $dateDebut = $request->request->get("datedebut");
            $dateFin = $request->request->get("datefin");

            if (empty($dateDebut) || empty($dateFin)) {
                return $this->redirectToRoute('app_reservation_voyage_index', [
                    'id' => $voyage->getId(),
                ], Response::HTTP_SEE_OTHER);
            }


            return $this->redirectToRoute('app_reservation_voyage_index_search', [
                'dateDebut' => $dateDebut,
                'dateFin' => $dateFin,
                'id' => $voyage->getId(),
                // 'voyages' => $voyages,
                //  'form' => $form,

            ], Response::HTTP_SEE_OTHER);


        }


        $totalChargeVoyage = $this->entityManager->createQueryBuilder()
            ->select('SUM(c.montant) as totalMontant')
            ->from(Charge::class, 'c')
//            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
//            ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
            ->andWhere('c.voyage = :val')
//            ->andWhere('c.dateCharge >= :today')
//            ->setParameter('today', $today->format('Y-m-d'))
            ->setParameter('val', $voyage->getId())
            ->getQuery()
            ->getSingleScalarResult();

        $totalTicketsVendusPrix = $this->entityManager->createQueryBuilder()
            ->select('SUM(t.prix) as totalPrix')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
            ->andWhere('v.id = :val')
//            ->andWhere('r.dateReservation = :today')
//            ->setParameter('today', $today->format('Y-m-d'))
            ->setParameter('val', $voyage->getId())
            ->getQuery()
            ->getSingleScalarResult();

        $totalTicketsVendus = $this->entityManager->createQueryBuilder()
            ->select('COUNT(t.id) as total')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
            ->andWhere('v.id = :val')
//            ->andWhere('r.dateReservation = :today')
//            ->setParameter('today', $today->format('Y-m-d'))
            ->setParameter('val', $voyage->getId())
            ->getQuery()
            ->getSingleScalarResult();

        return $this->render('reservation/indexVoyage.html.twig', [
            'reservations' => $reservations,
            'totalTicketsVendusPrix' => $totalTicketsVendusPrix,
            'totalTicketsVendus' => $totalTicketsVendus,
            'form' => $form,
            'voyage' => $voyage,
            'totalmontant' => $totalChargeVoyage,


        ]);
    }


    #[Route('/{id}/{dateDebut}/{dateFin}/index/search', name: 'app_reservation_voyage_index_search', methods: ['GET', 'POST'])]
    public function indexVoyageSearch(Request               $request,
                                      ReservationRepository $reservationRepository,
                                      UserSocieteRepository $userSocieteRepository,
                                      Voyage                $voyage,
                                                            $dateDebut,
                                                            $dateFin): Response
    {


        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $reservation = new Reservation();
        $form = $this->createForm(ReservationType::class, $reservation);
        $form->remove('paied');
        $form->remove('dateReservation');
        $form->remove('statut');
        $form->remove('billnumber');
        $form->remove('transit');
        $form->remove('client');
        $form->remove('societe');

        $form->handleRequest($request);

//        $reservations = $reservationRepository->findBy(array('societe' => $societe->getSociete(), 'statut' => 1));

        $today = new \DateTime();  // Récupère la date actuelle


        $reservations = $this->entityManager->getRepository(Reservation::class)
            ->createQueryBuilder('r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
            ->where('r.dateReservation BETWEEN :dateDebut AND :dateFin')
            ->andWhere('r.societe=:societe')
            ->andWhere('v.id = :val')
            ->setParameter('val', $voyage->getId())
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('societe', $societe->getSociete())
            ->getQuery()
            ->getResult();


        $totalChargeVoyage = $this->entityManager->createQueryBuilder()
            ->select('SUM(c.montant) as totalMontant')
            ->from(Charge::class, 'c')
//            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = c.voyage')
            ->andWhere('c.voyage = :val')
            ->andWhere('c.dateCharge BETWEEN :dateDebut AND :dateFin')
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('val', $voyage->getId())
            ->getQuery()
            ->getSingleScalarResult();


        $totalTicketsVendusPrix = $this->entityManager->createQueryBuilder()
            ->select('SUM(t.prix) as totalPrix')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
            ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
            ->andWhere('r.societe=:societe')
            ->andWhere('v.id = :val')
            ->setParameter('val', $voyage->getId())
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('societe', $societe->getSociete())
            ->getQuery()
            ->getSingleScalarResult();

        $totalTicketsVendus = $this->entityManager->createQueryBuilder()
            ->select('COUNT(t.id) as total')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
            ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
            ->andWhere('r.societe=:societe')
            ->andWhere('v.id = :val')
            ->setParameter('val', $voyage->getId())
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('societe', $societe->getSociete())
            ->getQuery()
            ->getSingleScalarResult();


        if ($form->isSubmitted() && $form->isValid()) {

            $dateDebut = $request->request->get("datedebut");
            $dateFin = $request->request->get("datefin");

            if (empty($dateDebut) || empty($dateFin)) {
                return $this->redirectToRoute('app_reservation_index', [
                    'id' => $voyage->getId(),
                ], Response::HTTP_SEE_OTHER);
            }


            return $this->redirectToRoute('app_reservation_index_search', [
                'dateDebut' => $dateDebut,
                'dateFin' => $dateFin,
                'id' => $voyage->getId(),
                'totalmontant' => $totalChargeVoyage,

                // 'voyages' => $voyages,
                //  'form' => $form,

            ], Response::HTTP_SEE_OTHER);


        }


        return $this->render('reservation/indexVoyage.html.twig', [
            'reservations' => $reservations,
            'totalTicketsVendusPrix' => $totalTicketsVendusPrix,
            'totalTicketsVendus' => $totalTicketsVendus,
            'form' => $form,
            'voyage' => $voyage,
            'totalmontant' => $totalChargeVoyage,


        ]);
    }


    #[Route('/as/voyage/user', name: 'app_reservation_voyage_index_user', methods: ['GET', 'POST'])]
    public function indexVoyageUser(Request               $request,
                                    ReservationRepository $reservationRepository,
                                    UserSocieteRepository $userSocieteRepository,
                                    VoyageRepository      $voyageRepository
    ): Response
    {

        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }


        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $reservation = new Reservation();
        $form = $this->createForm(ReservationType::class, $reservation);
        $form->remove('paied');
        $form->remove('dateReservation');
        $form->remove('statut');
        $form->remove('billnumber');
        $form->remove('transit');
        $form->remove('client');
        $form->remove('societe');

        $form->handleRequest($request);
//        $reservations = $reservationRepository->findByVoyage($voyage->getId());

        $today = new \DateTime();  // Récupère la date actuelle

        if ($this->isGranted('SUPER_ADMIN')) {

            $reservations = $voyageRepository->getVoyageDetailsAdmin($today->format('Y-m-d'));
        } else {
            $reservations = $voyageRepository->getVoyageDetails($this->getUser(), $today->format('Y-m-d'));

        }


        if ($form->isSubmitted() && $form->isValid()) {

            $dateDebut = $request->request->get("datedebut");
            $dateFin = $request->request->get("datefin");

            if (empty($dateDebut) || empty($dateFin)) {
                return $this->redirectToRoute('app_reservation_voyage_index_user', [
                    'id' => $voyage->getId(),
                ], Response::HTTP_SEE_OTHER);
            }


            return $this->redirectToRoute('app_reservation_voyage_index_search_user', [
                'dateDebut' => $dateDebut,
                'dateFin' => $dateFin,
                // 'voyages' => $voyages,
                //  'form' => $form,

            ], Response::HTTP_SEE_OTHER);


        }

        if ($this->isGranted('SUPER_ADMIN')) {

            $totalChargeVoyage = $this->entityManager->createQueryBuilder()
                ->select('SUM(c.montant) as totalMontant')
                ->from(Charge::class, 'c')
                ->andWhere('c.dateCharge >= :today')
                ->setParameter('today', $today->format('Y-m-d'))
                ->getQuery()
                ->getSingleScalarResult();

            $totalTicketsVendusPrix = $this->entityManager->createQueryBuilder()
                ->select('SUM(t.prix) as totalPrix')
                ->from(Reservation::class, 'r')
                ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
                ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
                ->andWhere('r.dateReservation >= :today')
                ->setParameter('today', $today->format('Y-m-d'))
                ->getQuery()
                ->getSingleScalarResult();

            $totalTicketsVendus = $this->entityManager->createQueryBuilder()
                ->select('COUNT(t.id) as total')
                ->from(Reservation::class, 'r')
                ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
                ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
                ->andWhere('r.dateReservation >= :today')
                ->setParameter('today', $today->format('Y-m-d'))
                ->getQuery()
                ->getSingleScalarResult();

        } else {
            $totalChargeVoyage = $this->entityManager->createQueryBuilder()
                ->select('SUM(c.montant) as totalMontant')
                ->from(Charge::class, 'c')
                ->andWhere('c.user = :user')
                ->andWhere('c.dateCharge >= :today')
                ->setParameter('user', $this->getUser())
                ->setParameter('today', $today->format('Y-m-d'))
                ->getQuery()
                ->getSingleScalarResult();

            $totalTicketsVendusPrix = $this->entityManager->createQueryBuilder()
                ->select('SUM(t.prix) as totalPrix')
                ->from(Reservation::class, 'r')
                ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
                ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
                ->andWhere('r.user = :user')
                ->andWhere('r.dateReservation >= :today')
                ->setParameter('user', $this->getUser())
                ->setParameter('today', $today->format('Y-m-d'))
                ->getQuery()
                ->getSingleScalarResult();

            $totalTicketsVendus = $this->entityManager->createQueryBuilder()
                ->select('COUNT(t.id) as total')
                ->from(Reservation::class, 'r')
                ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
                ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
                ->andWhere('r.user = :user')
                ->andWhere('r.dateReservation >= :today')
                ->setParameter('user', $this->getUser())
                ->setParameter('today', $today->format('Y-m-d'))
                ->getQuery()
                ->getSingleScalarResult();

        }

        return $this->render('reservation/indexVoyageUser.html.twig', [
            'reservations' => $reservations,
            'totalTicketsVendusPrix' => $totalTicketsVendusPrix,
            'totalTicketsVendus' => $totalTicketsVendus,
            'form' => $form,
            'totalmontant' => $totalChargeVoyage,


        ]);
    }


    #[Route('/as/{dateDebut}/{dateFin}/index/search/user/user/', name: 'app_reservation_voyage_index_search_user', methods: ['GET', 'POST'])]
    public function indexVoyageSearchUser(Request               $request,
                                          ReservationRepository $reservationRepository,
                                          UserSocieteRepository $userSocieteRepository,
                                          VoyageRepository      $voyageRepository,
                                                                $dateDebut,
                                                                $dateFin): Response
    {


        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $reservation = new Reservation();
        $form = $this->createForm(ReservationType::class, $reservation);
        $form->remove('paied');
        $form->remove('dateReservation');
        $form->remove('statut');
        $form->remove('billnumber');
        $form->remove('transit');
        $form->remove('client');
        $form->remove('societe');

        $form->handleRequest($request);

//        $reservations = $reservationRepository->findBy(array('societe' => $societe->getSociete(), 'statut' => 1));


        if ($this->isGranted('SUPER_ADMIN')) {

            $reservations = $voyageRepository->getVoyageDetailsSearchAdmin($dateDebut, $dateFin);

        } else {
            $reservations = $voyageRepository->getVoyageDetailsSearch($this->getUser(), $dateDebut, $dateFin);

        }


        /*     $reservations = $this->entityManager->getRepository(Reservation::class)
                 ->createQueryBuilder('r')
                 ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
                 ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
                 ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
                 ->andWhere('r.societe=:societe')
                 ->andWhere('v.id = :val')
                 ->setParameter('val', $voyage->getId())
                 ->setParameter('dateDebut', $dateDebut)
                 ->setParameter('dateFin', $dateFin)
                 ->setParameter('societe', $societe->getSociete())
                 ->getQuery()
                 ->getResult();*/

        if ($this->isGranted('SUPER_ADMIN')) {


            $totalTicketsVendusPrix = $this->entityManager->createQueryBuilder()
                ->select('SUM(t.prix) as totalPrix')
                ->from(Reservation::class, 'r')
                ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
                ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
                ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
                ->andWhere('r.societe=:societe')
                ->setParameter('dateDebut', $dateDebut)
                ->setParameter('dateFin', $dateFin)
                ->setParameter('societe', $societe->getSociete())
                ->getQuery()
                ->getSingleScalarResult();

            $totalTicketsVendus = $this->entityManager->createQueryBuilder()
                ->select('COUNT(t.id) as total')
                ->from(Reservation::class, 'r')
                ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
                ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
                ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
                ->andWhere('r.societe=:societe')
                ->setParameter('dateDebut', $dateDebut)
                ->setParameter('dateFin', $dateFin)
                ->setParameter('societe', $societe->getSociete())
                ->getQuery()
                ->getSingleScalarResult();
        } else {
            $totalTicketsVendusPrix = $this->entityManager->createQueryBuilder()
                ->select('SUM(t.prix) as totalPrix')
                ->from(Reservation::class, 'r')
                ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
                ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
                ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
                ->andWhere('r.societe=:societe')
                ->andWhere('r.user = :user')
                ->setParameter('user', $this->getUser())
                ->setParameter('dateDebut', $dateDebut)
                ->setParameter('dateFin', $dateFin)
                ->setParameter('societe', $societe->getSociete())
                ->getQuery()
                ->getSingleScalarResult();

            $totalTicketsVendus = $this->entityManager->createQueryBuilder()
                ->select('COUNT(t.id) as total')
                ->from(Reservation::class, 'r')
                ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
                ->Join('App\Entity\Voyage', 'v', 'WITH', 'v.id = t.voyage')
                ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
                ->andWhere('r.societe=:societe')
                ->andWhere('r.user = :user')
                ->setParameter('user', $this->getUser())
                ->setParameter('dateDebut', $dateDebut)
                ->setParameter('dateFin', $dateFin)
                ->setParameter('societe', $societe->getSociete())
                ->getQuery()
                ->getSingleScalarResult();
        }

        if ($form->isSubmitted() && $form->isValid()) {

            $dateDebut = $request->request->get("datedebut");
            $dateFin = $request->request->get("datefin");

            if (empty($dateDebut) || empty($dateFin)) {
                return $this->redirectToRoute('app_reservation_index_user', [
                ], Response::HTTP_SEE_OTHER);
            }


            return $this->redirectToRoute('app_reservation_index_search_user', [
                'dateDebut' => $dateDebut,
                'dateFin' => $dateFin,
                // 'voyages' => $voyages,
                //  'form' => $form,

            ], Response::HTTP_SEE_OTHER);


        }


        return $this->render('reservation/indexVoyageUser.html.twig', [
            'reservations' => $reservations,
            'totalTicketsVendusPrix' => $totalTicketsVendusPrix,
            'totalTicketsVendus' => $totalTicketsVendus,
            'form' => $form,
            'totalmontant' => 0,


        ]);
    }


    #[Route('/{id}/voyage/transit', name: 'app_reservation_tansit_index', methods: ['GET', 'POST'])]
    public function indexTransit(Request               $request,
                                 ReservationRepository $reservationRepository,
                                 UserSocieteRepository $userSocieteRepository,
                                 Transit               $transit): Response
    {


        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $reservation = new Reservation();
        $form = $this->createForm(ReservationType::class, $reservation);
        $form->remove('paied');
        $form->remove('dateReservation');
        $form->remove('statut');
        $form->remove('billnumber');
        $form->remove('transit');
        $form->remove('client');
        $form->remove('societe');

        $form->handleRequest($request);
        $reservations = $reservationRepository->findBy(array('transit' => $transit->getId()));


        if ($form->isSubmitted() && $form->isValid()) {

            $dateDebut = $request->request->get("datedebut");
            $dateFin = $request->request->get("datefin");

            if (empty($dateDebut) || empty($dateFin)) {
                return $this->redirectToRoute('app_reservation_tansit_index', [
                    'id' => $transit->getId(),
                ], Response::HTTP_SEE_OTHER);
            }


            return $this->redirectToRoute('app_reservation_transit_index_search', [
                'dateDebut' => $dateDebut,
                'dateFin' => $dateFin,
                'id' => $transit->getId(),
                // 'voyages' => $voyages,
                //  'form' => $form,

            ], Response::HTTP_SEE_OTHER);


        }

        $totalTicketsVendusPrix = $this->entityManager->createQueryBuilder()
            ->select('SUM(t.prix) as totalPrix')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->andWhere('t.id = :val')
            ->setParameter('val', $transit->getId())
            ->getQuery()
            ->getSingleScalarResult();

        $totalTicketsVendus = $this->entityManager->createQueryBuilder()
            ->select('COUNT(t.id) as total')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->andWhere('t.id = :val')
            ->setParameter('val', $transit->getId())
            ->getQuery()
            ->getSingleScalarResult();

        return $this->render('reservation/indexTransit.html.twig', [
            'reservations' => $reservations,
            'totalTicketsVendusPrix' => $totalTicketsVendusPrix,
            'totalTicketsVendus' => $totalTicketsVendus,
            'form' => $form,
            'voyage' => $transit


        ]);
    }


    #[Route('/{id}/{dateDebut}/{dateFin}/index/search', name: 'app_reservation_transit_index_search', methods: ['GET', 'POST'])]
    public function indexTransitSearch(Request               $request,
                                       ReservationRepository $reservationRepository,
                                       UserSocieteRepository $userSocieteRepository,
                                       Transit               $transit,
                                                             $dateDebut,
                                                             $dateFin): Response
    {


        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $reservation = new Reservation();
        $form = $this->createForm(ReservationType::class, $reservation);
        $form->remove('paied');
        $form->remove('dateReservation');
        $form->remove('statut');
        $form->remove('billnumber');
        $form->remove('transit');
        $form->remove('client');
        $form->remove('societe');

        $form->handleRequest($request);

//        $reservations = $reservationRepository->findBy(array('societe' => $societe->getSociete(), 'statut' => 1));

        $reservations = $this->entityManager->getRepository(Reservation::class)
            ->createQueryBuilder('r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
            ->andWhere('r.societe=:societe')
            ->andWhere('t.id = :val')
            ->setParameter('val', $transit->getId())
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('societe', $societe->getSociete())
            ->getQuery()
            ->getResult();


        $totalTicketsVendusPrix = $this->entityManager->createQueryBuilder()
            ->select('SUM(t.prix) as totalPrix')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
            ->andWhere('r.societe=:societe')
            ->andWhere('t.id = :val')
            ->setParameter('val', $transit->getId())
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('societe', $societe->getSociete())
            ->getQuery()
            ->getSingleScalarResult();

        $totalTicketsVendus = $this->entityManager->createQueryBuilder()
            ->select('COUNT(t.id) as total')
            ->from(Reservation::class, 'r')
            ->Join('App\Entity\Transit', 't', 'WITH', 't.id = r.transit')
            ->where('t.datedepart BETWEEN :dateDebut AND :dateFin')
            ->andWhere('r.societe=:societe')
            ->andWhere('t.id = :val')
            ->setParameter('val', $transit->getId())
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('societe', $societe->getSociete())
            ->getQuery()
            ->getSingleScalarResult();


        if ($form->isSubmitted() && $form->isValid()) {

            $dateDebut = $request->request->get("datedebut");
            $dateFin = $request->request->get("datefin");

            if (empty($dateDebut) || empty($dateFin)) {
                return $this->redirectToRoute('app_reservation_tansit_index', [
                    'id' => $transit->getId(),
                ], Response::HTTP_SEE_OTHER);
            }


            return $this->redirectToRoute('app_reservation_transit_index_search', [
                'dateDebut' => $dateDebut,
                'dateFin' => $dateFin,
                'id' => $transit->getId(),
                // 'voyages' => $voyages,
                //  'form' => $form,

            ], Response::HTTP_SEE_OTHER);


        }


        return $this->render('reservation/indexTransit.html.twig', [
            'reservations' => $reservations,
            'totalTicketsVendusPrix' => $totalTicketsVendusPrix,
            'totalTicketsVendus' => $totalTicketsVendus,
            'form' => $form,
            'voyage' => $transit,


        ]);
    }


//    ------------------------------------- AJOUTER UNE RESERVATION


    #[Route(path: '/{id}/1/info/1/t/i/paiement', name: 'app_reservation_info_paiement', methods: ['GET', 'POST'])]
    public function indexInfoPaiement(Request                $request, TransitRepository $transitRepository,
                                      Voyage                 $voyage,
                                      VoyageRepository       $voyageRepository,
                                      EntityManagerInterface $entityManager): Response
    {
        // --- Security
        if (!$this->isGranted('RESERVATION_CREATE') && !$this->isGranted('SUPER_ADMIN')) {
            throw $this->createAccessDeniedException();
        }

        // --- Transit (etape 1) pour ce voyage
        $transit = $transitRepository->findOneBy([
            'voyage' => $voyage->getId(),
            'numeroetape' => 1,
        ]);

        if (!$transit) {
            throw $this->createNotFoundException("Transit (étape 1) introuvable pour ce voyage.");
        }

        // --- Optionnel: si plus de place
        if ($voyage->getAvailable() !== null && $voyage->getAvailable() <= 0) {
            $this->addFlash('error', "Plus de sièges disponibles pour ce voyage.");
        }

        // --- Entities
        $client = new Client();
        $reservation = new Reservation();
        $reservation->setTransit($transit);

        // --- Form (Client + dropdown siege disponible)
        $request->getSession()->start();
        $form = $this->createForm(ClientType::class, $client, [
            'transit' => $transit, // IMPORTANT
        ]);
        $form->handleRequest($request);

        /*if ($form->isSubmitted()) {
            dump('submitted', $form->isValid());
            dump((string) $form->getErrors(true, false));
            dump($form->has('vehiculesiege') ? $form->get('vehiculesiege')->getData() : 'no field vehiculesiege');
            die;
        }*/


        if ($form->isSubmitted() && $form->isValid()) {

            /** @var VehiculeSiege|null $selectedSeat */
            $selectedSeat = $form->get('vehiculesiege')->getData();
            if (!$selectedSeat) {
                $this->addFlash('error', 'Veuillez choisir un siège.');
                return $this->render('reservation/infoPaiement.html.twig', [
                    'voyage' => $voyage,
                    'transit' => $transit,
                    'form' => $form->createView(),
                    'reservation' => $reservation,
                ]);
            }

            // --- Dates
            $today = new \DateTimeImmutable();
            $dateString = $today->format('dmY');

            // --- Préparer client
            $client->setSociete($transit->getVoyage()->getSociete());
            $client->setStatut(true); // si ton champ est bool (sinon remets 1)

            // --- Préparer reservation
            $reservation->setClient($client);
            $reservation->setSociete($voyage->getSociete());
            $reservation->setDateReservation(\DateTime::createFromImmutable($today)); // ton champ est DATE_MUTABLE
            $reservation->setStatut(true); // idem bool
            $reservation->setPaied(true);
            $reservation->setUser($this->getUser());
            $reservation->setVehiculesiege($selectedSeat);

            // --- Décrémenter available (garde-fou)
            if ($voyage->getAvailable() !== null) {
                if ($voyage->getAvailable() <= 0) {
                    $this->addFlash('error', "Plus de sièges disponibles.");
                    return $this->redirectToRoute('app_reservation_info_paiement', ['id' => $voyage->getId()]);
                }
                $voyage->setAvailable($voyage->getAvailable() - 1);
            }

            // --- Persist (1 seul flush idéalement)
            try {
                $entityManager->persist($client);
                $entityManager->persist($reservation);
                $entityManager->persist($voyage);
                $entityManager->flush(); // client id généré ici
            } catch (UniqueConstraintViolationException $e) {
                // Si tu as mis la contrainte UNIQUE(transit_id, vehiculesiege_id)
                $this->addFlash('error', "Ce siège vient d’être réservé. Choisis un autre siège.");
                return $this->redirectToRoute('app_reservation_info_paiement', ['id' => $voyage->getId()]);
            }

            // --- Post-flush : code client + billnumber (car besoin ID)
            $client->setCode('TD' . $client->getId());

            $billnumber = $transit->getVoyage()->getSociete()->getId()
                . $client->getId()
                . $dateString;

            $reservation->setBillnumber($billnumber);

            $entityManager->flush();

            return $this->redirectToRoute('app_reservation_success_paiement', [
                'transit' => $transit->getId(),
                'reservation' => $reservation->getId(),
            ], Response::HTTP_SEE_OTHER);
        }

        return $this->render('reservation/infoPaiement.html.twig', [
            'voyage' => $voyage,
            'transit' => $transit,
            'form' => $form->createView(),
            'reservation' => $reservation,
        ]);
    }


    /*
            //    ------------------------------------- AJOUTER UNE RESERVATION


            #[Route(path: '/{id}/1/info/1//i/paiement', name: 'app_reservation_info_paiement', methods: ['GET', 'POST'])]
            public function indexInfoPaiement(Request                $request, TransitRepository $transitRepository,
                                              Voyage                 $voyage,
                                              VoyageRepository       $voyageRepository,
                                              EntityManagerInterface $entityManager): Response
            {

                $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
                $superAmin = $this->isGranted('SUPER_ADMIN');

                if (!$canCreate && !$superAmin) {
                    throw $this->createAccessDeniedException();
                }

                $client = new Client();
                $reservation = new Reservation();
                $today = new \DateTime();
                $dateString = $today->format('dmY');

                $transit = $transitRepository->findOneBy(array('voyage' => $voyage->getId(), 'numeroetape' => 1));

                $voyages = $transitRepository->findBy(array('id' => $transit->getId()));

                $form = $this->createForm(ClientType::class, $client);
                $form->handleRequest($request);



                if ($form->isSubmitted() && $form->isValid()) {

                    $client->setSociete($transit->getVoyage()->getSociete());
                    $client->setStatut(1);
                    $entityManager->persist($client);
                    $entityManager->flush();

                    $codeclient = "TD" . '' . $client->getId();
                    $client->setCode($codeclient);
                    $entityManager->persist($client);
                    $entityManager->flush();


                    $billnumber = $transit->getVoyage()->getSociete()->getId() . '' . $client->getId() . '' . $dateString;
    //            $capacite= $transit->getVoyage()->getVehicule()->getCapacite();
            $disponibilite=$transit->getVoyage()->getAvailable();
            $available=$disponibilite-1;


            $reservation->setClient($client);
            $reservation->setTransit($transit);
            $reservation->setSociete($voyage->getSociete());
            $reservation->setDateReservation($today);
            $reservation->setStatut(1);
            $reservation->setBillnumber($billnumber);
            $reservation->setPaied(1);
            $reservation->setUser($this->getUser());

            $entityManager->persist($reservation);

            $voyage->setAvailable($available);
            $entityManager->persist($voyage);


            $entityManager->flush();


            return $this->redirectToRoute('app_reservation_success_paiement', [
                'transit' => $transit->getId(),
                'reservation' => $reservation->getId()


            ], Response::HTTP_SEE_OTHER);
        }


    //        $dobReconverted = \DateTime::createFromFormat('Y-m-d H:i:s', $date_a);


            // $detail = $transitRepository->findByEtape($transit->getNumeroetape(), $transit->getVoyage()->getId());


            return $this->render('reservation/infoPaiement.html.twig', [
                'voyage' => $voyage,
                'transit' => $transit,
                'form' => $form,
                'reservation' => $reservation

            ]);
        }*/


    #[Route(path: '{transit}/{reservation}/reservation/success/paiement/success/paiement', name: 'app_reservation_success_paiement')]
    public function indexSuccessPaiement(Transit $transit, Reservation $reservation): Response
    {

        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        $qrCode = QrCode::create($reservation->getBillnumber())
            ->setEncoding(new Encoding('UTF-8'))
            ->setErrorCorrectionLevel(ErrorCorrectionLevel::Low)
            ->setSize(200)
            ->setMargin(10)
            ->setRoundBlockSizeMode(RoundBlockSizeMode::Margin)
            ->setForegroundColor(new Color(0, 0, 0))
            ->setBackgroundColor(new Color(255, 255, 255));
        $qrCodeData = $qrCode->getData(); // Assuming you have generateData
        $writer = new PngWriter();
        $result = $writer->write($qrCode);

        // Access the data string from the PngResult object
        $qrCodeImage = $result->getString(); // Replace with appropriate method

        $qrCodeImage = base64_encode($qrCodeImage);


        /*
                $writer = new PngWriter();

        // Create QR code
                $qrCode = QrCode::create('Life is too short to be generating QR codes')
                    ->setEncoding(new Encoding('UTF-8'))
                    ->setErrorCorrectionLevel(ErrorCorrectionLevel::Low)
                    ->setSize(300)
                    ->setMargin(10)
                    ->setRoundBlockSizeMode(RoundBlockSizeMode::Margin)
                    ->setForegroundColor(new Color(0, 0, 0))
                    ->setBackgroundColor(new Color(255, 255, 255));

        // Create generic logo
                $logo = Logo::create(__DIR__.'/assets/symfony.png')
                    ->setResizeToWidth(50)
                    ->setPunchoutBackground(true)
                ;

        // Create generic label
                $label = Label::create('Label')
                    ->setTextColor(new Color(255, 0, 0));

                $result = $writer->write($qrCode, $logo, $label);

        // Validate the result
                $writer->validateResult($result, 'Life is too short to be generating QR codes');

        // Save it to a file
                $result->saveToFile(__DIR__.'/assets/qrcode/qrcode.png');

                $dataUri = $result->getDataUri();*/


        return $this->render('reservation/successPaiement.html.twig', [
            'voyage' => $transit,
            'reservation' => $reservation,
            'qr_code_image' => $qrCodeImage,
            'qrCodeImage' => $qrCodeImage,
        ]);

    }


    #[Route(path: '{transit}/{reservation}/qr/qr/qr/reservation/success/paiement/success/paiement', name: 'app_reservation_success_paiement_impression')]
    public function imprimerSuccessPaiement(Transit $transit, Reservation $reservation): Response
    {

        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        /* $qrCode = QrCode::create($reservation->getBillnumber())
             ->setEncoding(new Encoding('UTF-8'))
             ->setErrorCorrectionLevel(ErrorCorrectionLevel::Low)
             ->setSize(200)
             ->setMargin(10)
             ->setRoundBlockSizeMode(RoundBlockSizeMode::Margin)
             ->setForegroundColor(new Color(0, 0, 0))
             ->setBackgroundColor(new Color(255, 255, 255));
         $qrCodeData = $qrCode->getData(); // Assuming you have generateData
         $writer = new PngWriter();
         $result = $writer->write($qrCode);

         // Access the data string from the PngResult object
         $qrCodeImage = $result->getString(); // Replace with appropriate method

         $qrCodeImage = base64_encode($qrCodeImage);*/


        return $this->render('reservation/impressionFacture.html.twig', [
            'voyage' => $transit,
            'reservation' => $reservation,
//            'qr_code_image' => $qrCodeImage,
//            'qrCodeImage' => $qrCodeImage,
        ]);

    }


    #[Route('/new', name: 'app_reservation_new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $entityManager): Response
    {

        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $superAmin = $this->isGranted('SUPER_ADMIN');

        if (!$canCreate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $reservation = new Reservation();
        $form = $this->createForm(ReservationType::class, $reservation);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->persist($reservation);
            $entityManager->flush();

            return $this->redirectToRoute('app_reservation_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('reservation/new.html.twig', [
            'reservation' => $reservation,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'app_reservation_show', methods: ['GET'])]
    public function show(Reservation $reservation): Response
    {
        $canView = $this->isGranted('RESERVATION_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('RESERVATION_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        return $this->render('reservation/show.html.twig', [
            'reservation' => $reservation,
        ]);
    }

    #[Route('/{id}/edit', name: 'app_reservation_edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Reservation $reservation, EntityManagerInterface $entityManager): Response
    {
        $canUpdate = $this->isGranted('RESERVATION_UPDATE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canUpdate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $form = $this->createForm(ReservationType::class, $reservation);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->flush();

            return $this->redirectToRoute('app_reservation_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('reservation/edit.html.twig', [
            'reservation' => $reservation,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'app_reservation_delete', methods: ['POST'])]
    public function delete(Request $request, Reservation $reservation, EntityManagerInterface $entityManager): Response
    {

        $canDelete = $this->isGranted('RESERVATION_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        if ($this->isCsrfTokenValid('delete' . $reservation->getId(), $request->getPayload()->get('_token'))) {
            $entityManager->remove($reservation);
            $entityManager->flush();
        }

        return $this->redirectToRoute('app_reservation_index', [], Response::HTTP_SEE_OTHER);
    }
}
