<?php

namespace App\Controller;

use App\Entity\Role;
use App\Entity\RolePermission;
use App\Form\Model\RolePermissionDTO;
use App\Form\RolePermissionForm;
use App\Form\RolePermissionMassFormType;
use App\Form\RolePermissionRoleForm;
use App\Repository\PermissionRepository;
use App\Repository\RolePermissionRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/role/permission')]
final class RolePermissionController extends AbstractController
{
    #[Route(name: 'app_role_permission_index', methods: ['GET'])]
    public function index(RolePermissionRepository $rolePermissionRepository): Response
    {
        return $this->render('role_permission/index.html.twig', [
            'role_permissions' => $rolePermissionRepository->findAll(),
        ]);
    }

    #[Route('role/{id}/role', name: 'app_role_permission_index_role', methods: ['GET'])]
    public function indexRole(RolePermissionRepository $rolePermissionRepository,
                              Role                     $role): Response
    {
        return $this->render('role_permission/indexRole.html.twig', [
            'role_permissions' => $rolePermissionRepository->findBy(array('role' => $role)),
            'role' => $role
        ]);
    }


    #[Route('/new', name: 'app_role_permission_new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $entityManager): Response
    {
        $rolePermission = new RolePermission();
        $form = $this->createForm(RolePermissionForm::class, $rolePermission);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->persist($rolePermission);
            $entityManager->flush();

            return $this->redirectToRoute('app_role_permission_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('role_permission/new.html.twig', [
            'role_permission' => $rolePermission,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/form', name: 'app_role_permission_form_partial', methods: ['GET'])]
    public function getFormPartial(
        Role                     $role,
        RolePermissionRepository $rolePermissionRepository
    ): Response
    {
        $existingLinks = $rolePermissionRepository->findBy(['role' => $role]);
        $selectedPermissions = array_map(fn($rp) => $rp->getPermission(), $existingLinks);

        $dto = new RolePermissionDTO();
        $dto->setPermissions($selectedPermissions);

        $form = $this->createForm(RolePermissionMassFormType::class, $dto, [
            'action' => $this->generateUrl('app_role_permission_submit_ajax', ['id' => $role->getId()]),
            'method' => 'POST'
        ]);

        return $this->render('role_permission/_form_ajax.html.twig', [
            'form' => $form->createView(),
            'role' => $role
        ]);
    }


    #[Route('/{id}/submit', name: 'app_role_permission_submit_ajax', methods: ['POST'])]
    public function submitAjax(
        Role                     $role,
        Request                  $request,
        RolePermissionRepository $rolePermissionRepository,
        EntityManagerInterface   $em
    ): Response
    {
        $dto = new RolePermissionDTO();
        $form = $this->createForm(RolePermissionMassFormType::class, $dto);
        $form->handleRequest($request);

        if ($form->isSubmitted() && !$form->isValid()) {
            foreach ($form->getErrors(true) as $error) {
                dump($error->getMessage());
            }
            dd('Formulaire invalide');
        }

        $selectedPermissions = $dto->getPermissions();
        $existingLinks = $rolePermissionRepository->findBy(['role' => $role]);

        foreach ($existingLinks as $rp) {
            if (!in_array($rp->getPermission(), $selectedPermissions, true)) {
                $em->remove($rp);
            }
        }

        foreach ($selectedPermissions as $permission) {
            $exists = $rolePermissionRepository->findOneBy([
                'role' => $role,
                'permission' => $permission,
            ]);

            if (!$exists) {
                $rp = new RolePermission();
                $rp->setRole($role);
                $rp->setPermission($permission);
                $em->persist($rp);
            }
        }

        $em->flush();
        return new Response('OK');
    }




    #[Route('/{id}/table', name: 'app_role_permission_partial_table', methods: ['GET'])]
    public function tablePartial(RolePermissionRepository $repo, Role $role): Response
    {
        return $this->render('role_permission/_table_wrapper.html.twig', [
            'role_permissions' => $repo->findBy(['role' => $role]),
            'role' => $role,
        ]);
    }


    #[Route('/{id}/testtable', name: 'app_role_permission_test_table', methods: ['GET'])]
    public function testTable(Role $role, RolePermissionRepository $repo): Response
    {
        return $this->render('role_permission/_test_table_body.html.twig', [
            'role_permissions' => $repo->findBy(['role' => $role])
        ]);
    }



    #[Route('/edit/{id}/for/role', name: 'app_role_permission_edit_for_role', methods: ['GET', 'POST'])]
    public
    function editPermissionsForRole(
        Role                     $role,
        Request                  $request,
        EntityManagerInterface   $entityManager,
        RolePermissionRepository $rolePermissionRepository,
        PermissionRepository     $permissionRepository
    ): Response
    {
        // Récupérer les permissions déjà liées au rôle
        $existingLinks = $rolePermissionRepository->findBy(['role' => $role]);
        $existingPermissions = array_map(fn($rp) => $rp->getPermission(), $existingLinks);

        // Préparer le DTO avec les permissions pré-cochées
        $dto = new RolePermissionDTO();
        $dto->setPermissions($existingPermissions);

        // Créer le formulaire
        $form = $this->createForm(RolePermissionMassFormType::class, $dto);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $selectedPermissions = $dto->getPermissions();

            // Supprimer les permissions décochées
            foreach ($existingLinks as $rolePermission) {
                if (!in_array($rolePermission->getPermission(), $selectedPermissions, true)) {
                    $entityManager->remove($rolePermission);
                }
            }

            // Ajouter les nouvelles permissions cochées
            foreach ($selectedPermissions as $permission) {
                $exists = $rolePermissionRepository->findOneBy([
                    'role' => $role,
                    'permission' => $permission,
                ]);

                if (!$exists) {
                    $new = new RolePermission();
                    $new->setRole($role);
                    $new->setPermission($permission);
                    $entityManager->persist($new);
                }
            }

            $entityManager->flush();

            $this->addFlash('success', 'Permissions mises à jour avec succès.');
            return $this->redirectToRoute('app_role_permission_index_role', [
                'id' => $role->getId(),

            ], Response::HTTP_SEE_OTHER);

        }

        return $this->render('role_permission/newRole.html.twig', [
            'form' => $form->createView(),
            'role' => $role,
        ]);
    }

    #[Route('/{id}', name: 'app_role_permission_show', methods: ['GET'])]
    public
    function show(RolePermission $rolePermission): Response
    {
        return $this->render('role_permission/show.html.twig', [
            'role_permission' => $rolePermission,
        ]);
    }

    #[Route('/{id}/edit', name: 'app_role_permission_edit', methods: ['GET', 'POST'])]
    public
    function edit(Request $request, RolePermission $rolePermission, EntityManagerInterface $entityManager): Response
    {
        $form = $this->createForm(RolePermissionForm::class, $rolePermission);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->flush();

            return $this->redirectToRoute('app_role_permission_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('role_permission/edit.html.twig', [
            'role_permission' => $rolePermission,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'app_role_permission_delete', methods: ['POST'])]
    public
    function delete(Request $request, RolePermission $rolePermission, EntityManagerInterface $entityManager): Response
    {
        if ($this->isCsrfTokenValid('delete' . $rolePermission->getId(), $request->getPayload()->getString('_token'))) {
            $entityManager->remove($rolePermission);
            $entityManager->flush();
        }

        return $this->redirectToRoute('app_role_permission_index', [], Response::HTTP_SEE_OTHER);
    }
}
