<?php

namespace App\Controller;

use App\Entity\Role;
use App\Entity\RolePermission;
use App\Entity\User;
use App\Entity\UserRole;
use App\Form\Model\RolePermissionDTO;
use App\Form\Model\UserRoleDTO;
use App\Form\RolePermissionMassFormType;
use App\Form\UserRoleForm;
use App\Form\UserRoleMassForm;
use App\Repository\PermissionRepository;
use App\Repository\PersonnelRepository;
use App\Repository\RolePermissionRepository;
use App\Repository\UserRepository;
use App\Repository\UserRoleRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/user/role')]
final class UserRoleController extends AbstractController
{
    #[Route(name: 'app_user_role_index', methods: ['GET'])]
    public function index(UserRoleRepository $userRoleRepository): Response
    {
        $canView = $this->isGranted('USERROLE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('USERROLE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('USERROLE_UPDATE');
        $canDelete = $this->isGranted('USERROLE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        return $this->render('user_role/index.html.twig', [
            'user_roles' => $userRoleRepository->findAll(),
        ]);
    }

    #[Route('/new', name: 'app_user_role_new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $entityManager): Response
    {
        $canCreate = $this->isGranted('USERROLE_CREATE');
        $superAmin = $this->isGranted('SUPER_ADMIN');
// ou $this->isGranted('VOYAGE_LIST', $subject)

        if (!$canCreate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        $userRole = new UserRole();
        $form = $this->createForm(UserRoleForm::class, $userRole);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->persist($userRole);
            $entityManager->flush();

            return $this->redirectToRoute('app_user_role_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('user_role/new.html.twig', [
            'user_role' => $userRole,
            'form' => $form,
        ]);
    }


    #[Route('/{id}/form', name: 'app_user_role_form_partial', methods: ['GET'])]
    public function getFormPartial(
        User               $user,
        UserRoleRepository $userRoleRepository
    ): Response
    {
//        dd("ooookkkkoo");

        $existingLinks = $userRoleRepository->findBy(['user' => $user]);
        $selectedRoles = array_map(fn($rp) => $rp->getRole(), $existingLinks);

        $dto = new UserRoleDTO();
        $dto->setRoles($selectedRoles);

        $form = $this->createForm(UserRoleMassForm::class, $dto, [
            'action' => $this->generateUrl('app_user_role_submit_ajax', ['id' => $user->getId()]),
            'method' => 'POST'
        ]);

        return $this->render('personnel/_form_ajax.html.twig', [
            'form' => $form->createView(),
            'user' => $user
        ]);
    }


    #[Route('/{id}/submit', name: 'app_user_role_submit_ajax', methods: ['POST'])]
    public function submitAjax(
        User                   $user,
        Request                $request,
        UserRoleRepository     $userRoleRepository,
        EntityManagerInterface $em
    ): Response
    {
        $dto = new UserRoleDTO();
        $form = $this->createForm(UserRoleMassForm::class, $dto);
        $form->handleRequest($request);

        if ($form->isSubmitted() && !$form->isValid()) {
            foreach ($form->getErrors(true) as $error) {
                dump($error->getMessage());
            }
            dd('Formulaire invalide');
        }

        $selectedRoles = $dto->getRoles();
        $existingLinks = $userRoleRepository->findBy(['user' => $user]);

        foreach ($existingLinks as $rp) {
            if (!in_array($rp->getRole(), $selectedRoles, true)) {
                $em->remove($rp);
            }
        }

        foreach ($selectedRoles as $role) {
            $exists = $userRoleRepository->findOneBy([
                'user' => $user,
                'role' => $role,
            ]);

            if (!$exists) {
                $rp = new UserRole();
                $rp->setUser($user);
                $rp->setRole($role);
                $em->persist($rp);
            }
        }

        $em->flush();
        return new Response('OK');
    }


    #[Route('/edit/{id}/for/role', name: 'app_user_role_edit_for_role', methods: ['GET', 'POST'])]
    public
    function editRoleforUser(
        User                   $user,
        UserRoleRepository     $userRoleRepository,
        Request                $request,
        EntityManagerInterface $entityManager,
        PersonnelRepository    $personnelRepository
    ): Response
    {

        $personnel = $personnelRepository->findOneBy(array('user' => $user));

        // Récupérer les permissions déjà liées au rôle
        $existingLinks = $userRoleRepository->findBy(['user' => $user]);
        $existingRoles = array_map(fn($rp) => $rp->getRole(), $existingLinks);

        // Préparer le DTO avec les permissions pré-cochées
        $dto = new UserRoleDTO();
        $dto->setRoles($existingRoles);

        // Créer le formulaire
        $form = $this->createForm(UserRoleMassForm::class, $dto);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $selectedRoles = $dto->getRoles();

            // Supprimer les permissions décochées
            foreach ($existingLinks as $userRoles) {
                if (!in_array($userRoles->getRole(), $selectedRoles, true)) {
                    $entityManager->remove($userRoles);
                }
            }

            // Ajouter les nouvelles permissions cochées
            foreach ($selectedRoles as $role) {
                $exists = $userRoleRepository->findOneBy([
                    'user' => $user,
                    'role' => $role,
                ]);

                if (!$exists) {
                    $new = new UserRole();
                    $new->setUser($user);
                    $new->setRole($role);
                    $entityManager->persist($new);
                }
            }

            $entityManager->flush();

            $this->addFlash('success', 'Role mises à jour avec succès.');
            return $this->redirectToRoute('app_user_role_show', [
                'id' => $personnel->getId(),

            ], Response::HTTP_SEE_OTHER);

        }

        return $this->render('personnel/newRole.html.twig', [
            'form' => $form->createView(),
            'user' => $user,
        ]);
    }



    #[Route('/{id}/table', name: 'app_user_role_partial_table', methods: ['GET'])]
    public function tablePartial(UserRoleRepository $repo, User $user): Response
    {
        return $this->render('personnel/_table_wrapper.html.twig', [
            'user_roles' => $repo->findBy(['user' => $user]),
            'user' => $user,
        ]);
    }




    #[Route('/{id}', name: 'app_user_role_show', methods: ['GET'])]
    public function show(UserRole $userRole): Response
    {
        $canView = $this->isGranted('USERROLE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('USERROLE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('USERROLE_UPDATE');
        $canDelete = $this->isGranted('USERROLE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        return $this->render('user_role/show.html.twig', [
            'user_role' => $userRole,
        ]);
    }

    #[Route('/{id}/edit', name: 'app_user_role_edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, UserRole $userRole, EntityManagerInterface $entityManager): Response
    {
        // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('USERROLE_UPDATE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canUpdate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $form = $this->createForm(UserRoleForm::class, $userRole);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->flush();

            return $this->redirectToRoute('app_user_role_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('user_role/edit.html.twig', [
            'user_role' => $userRole,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'app_user_role_delete', methods: ['POST'])]
    public function delete(Request $request, UserRole $userRole, EntityManagerInterface $entityManager): Response
    {

        $canDelete = $this->isGranted('USERROLE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        if ($this->isCsrfTokenValid('delete' . $userRole->getId(), $request->getPayload()->getString('_token'))) {
            $entityManager->remove($userRole);
            $entityManager->flush();
        }

        return $this->redirectToRoute('app_user_role_index', [], Response::HTTP_SEE_OTHER);
    }
}
