<?php

namespace App\Controller;

use App\Entity\TypeVehicule;
use App\Entity\Vehicule;
use App\Entity\VehiculeSiege;
use App\Form\VehiculeType;
use App\Repository\UserSocieteRepository;
use App\Repository\VehiculeRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/vehicule')]
class VehiculeController extends AbstractController
{
    #[Route('/{id}', name: 'app_vehicule_index', methods: ['GET'])]
    public function index(VehiculeRepository $vehiculeRepository,
                          TypeVehicule       $typeVehicule): Response
    {
        $canView = $this->isGranted('VEHICULE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('VEHICULE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('VEHICULE_UPDATE');
        $canDelete = $this->isGranted('VEHICULE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        return $this->render('vehicule/index.html.twig', [
            'vehicules' => $vehiculeRepository->findBy(array('typevehicule' => $typeVehicule->getId(), 'statut' => 1)),
            'typeVehicule' => $typeVehicule,
        ]);
    }

    #[Route('/societe/index', name: 'app_vehicule_societe_index', methods: ['GET'])]
    public function indexSociete(VehiculeRepository    $vehiculeRepository,
                                 UserSocieteRepository $userSocieteRepository,
    ): Response
    {

        $canView = $this->isGranted('VEHICULE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('VEHICULE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('VEHICULE_UPDATE');
        $canDelete = $this->isGranted('VEHICULE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));

        return $this->render('vehicule/indexSociete.html.twig', [
            'vehicules' => $vehiculeRepository->findBy(array('societe' => $societe->getSociete(), 'statut' => 1)),
        ]);
    }

    #[Route('/new/{id}', name: 'app_vehicule_new', methods: ['GET', 'POST'])]
    public function new(Request                $request,
                        EntityManagerInterface $entityManager,
                        TypeVehicule           $typeVehicule,
                        UserSocieteRepository  $userSocieteRepository): Response
    {
        $canCreate = $this->isGranted('VEHICULE_CREATE');
        $superAmin = $this->isGranted('SUPER_ADMIN');
// ou $this->isGranted('VOYAGE_LIST', $subject)

        if (!$canCreate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));


        $vehicule = new Vehicule();
        $form = $this->createForm(VehiculeType::class, $vehicule);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $vehicule->setTypevehicule($typeVehicule);
            $vehicule->setStatut(1);
            $vehicule->setSociete($societe->getSociete());
            $entityManager->persist($vehicule);


            $capacite = (int) $vehicule->getCapacite();
            for ($i = 1; $i <= $capacite; $i++) {
                $siege = new VehiculeSiege();
                $siege->setNumerosiege($i);
                $vehicule->addVehiculeSiege($siege); // cascade persist fera le job
            }



            $entityManager->flush();

            return $this->redirectToRoute('app_vehicule_index', [
                'id' => $typeVehicule->getId(),
            ], Response::HTTP_SEE_OTHER);
        }

        $response = new Response(null, $form->isSubmitted() ? 422 : 200);

        return $this->render('vehicule/new.html.twig', [
            'vehicule' => $vehicule,
            'form' => $form,
            'typeVehicule' => $typeVehicule,
        ], $response);
    }

    #[Route('/{id}', name: 'app_vehicule_show', methods: ['GET'])]
    public function show(Vehicule $vehicule): Response
    {
        $canView = $this->isGranted('VEHICULE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('VEHICULE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('VEHICULE_UPDATE');
        $canDelete = $this->isGranted('VEHICULE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }


        return $this->render('vehicule/show.html.twig', [
            'vehicule' => $vehicule,
        ]);
    }

    #[Route('/{id}/edit', name: 'app_vehicule_edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Vehicule $vehicule, EntityManagerInterface $entityManager): Response
    {

        $canUpdate = $this->isGranted('VEHICULE_UPDATE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canUpdate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $form = $this->createForm(VehiculeType::class, $vehicule);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->flush();

            return $this->redirectToRoute('app_vehicule_index', [
                'id'=>$vehicule->getTypevehicule()->getId(),
            ], Response::HTTP_SEE_OTHER);
        }

        return $this->render('vehicule/edit.html.twig', [
            'vehicule' => $vehicule,
            'form' => $form,
        ]);
    }


    #[Route('/{id}/edit/edit', name: 'app_ville_edit_delete', methods: ['GET', 'POST'])]
    public function editDelete(Request $request, Vehicule $vehicule, EntityManagerInterface $entityManager): Response
    {

        $canDelete = $this->isGranted('VEHICULE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $vehicule->setStatut(false);
        $entityManager->flush();

        return $this->redirectToRoute('app_vehicule_index', [
            'id' => $vehicule->getTypevehicule()->getId(),
        ], Response::HTTP_SEE_OTHER);

    }


    #[Route('/{id}', name: 'app_vehicule_delete', methods: ['POST'])]
    public function delete(Request $request, Vehicule $vehicule, EntityManagerInterface $entityManager): Response
    {

        $canDelete = $this->isGranted('VEHICULE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }


        if ($this->isCsrfTokenValid('delete' . $vehicule->getId(), $request->getPayload()->get('_token'))) {
            $entityManager->remove($vehicule);
            $entityManager->flush();
        }

        return $this->redirectToRoute('app_vehicule_index', [], Response::HTTP_SEE_OTHER);
    }
}
