<?php

namespace App\Controller;

use App\Entity\Province;
use App\Entity\Ville;
use App\Form\VilleType;
use App\Repository\VilleRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/ville')]
class VilleController extends AbstractController
{
    #[Route('/{id}', name: 'app_ville_index', methods: ['GET'])]
    public function index(VilleRepository $villeRepository, Province $province): Response
    {
        $canView = $this->isGranted('VILLE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('VILLE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('VILLE_UPDATE');
        $canDelete = $this->isGranted('VILLE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        return $this->render('ville/index.html.twig', [
            'villes' => $villeRepository->findBy(array('province' => $province->getId(),'statut'=>1)),
            'province' => $province
        ]);
    }

    #[Route('/new/{id}', name: 'app_ville_new', methods: ['GET', 'POST'])]
    public function new(Request                $request,
                        EntityManagerInterface $entityManager,
                        Province               $province): Response
    {
        $canCreate = $this->isGranted('VILLE_CREATE');
        $superAmin = $this->isGranted('SUPER_ADMIN');
// ou $this->isGranted('VOYAGE_LIST', $subject)

        if (!$canCreate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        $ville = new Ville();
        $form = $this->createForm(VilleType::class, $ville);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $ville->setProvince($province);
            $ville->setSociete($province->getSociete());
            $ville->setStatut(1);
            $entityManager->persist($ville);
            $entityManager->flush();

            return $this->redirectToRoute('app_ville_index', [
                'id'=>$province->getId(),
            ], Response::HTTP_SEE_OTHER);
        }

        $response = new Response(null, $form->isSubmitted() ? 422 : 200);

        return $this->render('ville/new.html.twig', [
            'ville' => $ville,
            'form' => $form,
            'province'=>$province,
        ], $response);
    }

    #[Route('/{id}', name: 'app_ville_show', methods: ['GET'])]
    public function show(Ville $ville): Response
    {
        $canView = $this->isGranted('VILLE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('VILLE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('VILLE_UPDATE');
        $canDelete = $this->isGranted('VILLE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        return $this->render('ville/show.html.twig', [
            'ville' => $ville,
        ]);
    }

    #[Route('/{id}/edit', name: 'app_ville_edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Ville $ville, EntityManagerInterface $entityManager): Response
    {


        $canUpdate = $this->isGranted('VILLE_UPDATE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canUpdate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $form = $this->createForm(VilleType::class, $ville);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->flush();

            return $this->redirectToRoute('app_ville_index', [
                'id'=>$ville->getProvince()->getId(),
            ], Response::HTTP_SEE_OTHER);
        }

        return $this->render('ville/edit.html.twig', [
            'ville' => $ville,
            'form' => $form,
        ]);
    }


    #[Route('/{id}/edit/edit', name: 'app_ville_edit_delete', methods: ['GET', 'POST'])]
    public function editDelete(Request $request, Ville $ville, EntityManagerInterface $entityManager): Response
    {

        $canDelete = $this->isGranted('VILLE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if ( !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        $ville->setStatut(false);
        $entityManager->flush();

        return $this->redirectToRoute('app_ville_index', [
            'id'=>$ville->getProvince()->getId(),
        ], Response::HTTP_SEE_OTHER);

    }


    #[Route('/{id}', name: 'app_ville_delete', methods: ['POST'])]
    public function delete(Request $request, Ville $ville, EntityManagerInterface $entityManager): Response
    {


        $canDelete = $this->isGranted('VILLE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        if ($this->isCsrfTokenValid('delete' . $ville->getId(), $request->getPayload()->get('_token'))) {
            $entityManager->remove($ville);
            $entityManager->flush();
        }

        return $this->redirectToRoute('app_ville_index', [], Response::HTTP_SEE_OTHER);
    }
}
