<?php

namespace App\Controller;

use App\Entity\Chauffeur;
use App\Entity\ChauffeurVoyage;
use App\Entity\Classe;
use App\Entity\Transit;
use App\Entity\Vehicule;
use App\Entity\Ville;
use App\Entity\Voyage;
use App\Form\ChauffeurVoyageType;
use App\Form\TransitType;
use App\Form\VoyageType;
use App\Form\VoyageTypee;
use App\Repository\UserSocieteRepository;
use App\Repository\VoyageRepository;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityRepository;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Doctrine\ORM\QueryBuilder;
use Symfony\Component\Form\Extension\Core\Type\CollectionType;
use function PHPUnit\Framework\isEmpty;


#[Route('/voyage')]
class VoyageController extends AbstractController
{

    private $entityManager;

    public function __construct(EntityManagerInterface $entityManager)
    {
        $this->entityManager = $entityManager;
    }

    #[Route('/', name: 'app_voyage_index', methods: ['GET', 'POST'])]
    public function index(Request               $request,
                          VoyageRepository      $voyageRepository,
                          UserSocieteRepository $userSocieteRepository): Response
    {


        $canView = $this->isGranted('VOYAGE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('VOYAGE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('VOYAGE_UPDATE');
        $canDelete = $this->isGranted('VOYAGE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $voyage = new Voyage();
        $form = $this->createForm(VoyageType::class, $voyage);
        $form->remove('datedepart');
        $form->remove('datearrive');
        $form->remove('reference');
        $form->remove('heuredepart');
        $form->remove('heurearrivee');

        $form->handleRequest($request);
        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));

        $voyages = $voyageRepository->findBy(array('societe' => $societe->getSociete()));


        if ($form->isSubmitted() && $form->isValid()) {

            $dateDebut = $request->request->get("datedebut");
            $dateFin = $request->request->get("datefin");

            if (empty($dateDebut) || empty($dateFin)) {
                return $this->redirectToRoute('app_voyage_index', [], Response::HTTP_SEE_OTHER);
            }


            return $this->redirectToRoute('app_voyage_index_search', [
                'dateDebut' => $dateDebut,
                'dateFin' => $dateFin,
                // 'voyages' => $voyages,
                //  'form' => $form,

            ], Response::HTTP_SEE_OTHER);


        }


        return $this->render('voyage/index.html.twig', [
            'voyages' => $voyages,
            'form' => $form,
            'testeur' => 0,


        ]);
    }


    #[Route('/{dateDebut}/{dateFin}/index/search', name: 'app_voyage_index_search', methods: ['GET', 'POST'])]
    public function indexVoyageSearch(Request               $request,
                                      VoyageRepository      $voyageRepository,
                                      UserSocieteRepository $userSocieteRepository,
                                                            $dateDebut,
                                                            $dateFin): Response
    {

        $canView = $this->isGranted('VOYAGE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('VOYAGE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('VOYAGE_UPDATE');
        $canDelete = $this->isGranted('VOYAGE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }
        $voyage = new Voyage();
        $form = $this->createForm(VoyageType::class, $voyage);
        $form->remove('datedepart');
        $form->remove('datearrive');
        $form->remove('reference');
        $form->remove('heuredepart');
        $form->remove('heurearrivee');

        $form->handleRequest($request);
        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));

        //    $voyages = $voyageRepository->findBy(array('societe' => $societe->getSociete()));


        $voyages = $this->entityManager->getRepository(Voyage::class)
            ->createQueryBuilder('v')
            ->where('v.datedepart BETWEEN :dateDebut AND :dateFin')
            ->andWhere('v.societe=:societe')
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->setParameter('societe', $societe->getSociete())
            ->getQuery()
            ->getResult();

        if ($form->isSubmitted() && $form->isValid()) {

            $dateDebuts = $request->request->get("datedebut");
            $dateFins = $request->request->get("datefin");

            if (empty($dateDebut) || empty($dateFin)) {
                return $this->redirectToRoute('app_voyage_index', [], Response::HTTP_SEE_OTHER);
            }

            return $this->redirectToRoute('app_voyage_index_search', [
                'dateDebut' => $dateDebut,
                'dateFin' => $dateFin,

            ], Response::HTTP_SEE_OTHER);

        }


        return $this->render('voyage/index.html.twig', [
            'voyages' => $voyages,
            'form' => $form,
            'debut' => $dateDebut,
            'fin' => $dateFin,
            'testeur' => 1,

        ]);
    }


    #[Route('/new', name: 'app_voyage_new', methods: ['GET', 'POST'])]
    public function new(Request                $request,
                        EntityManagerInterface $entityManager,
                        UserSocieteRepository  $userSocieteRepository): Response
    {


        $canCreate = $this->isGranted('VOYAGE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canCreate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }


        $societe = $userSocieteRepository->findOneBy(array('user' => $this->getUser()));

        $voyage = new Voyage();
        $transit = new Transit();

        $form = $this->createForm(VoyageType::class, $voyage)
            ->add('vehicule', EntityType::class, [
                'class' => Vehicule::class,
                'choice_label' => function ($vehicule) {
                    return $vehicule->getMarque() . ' ' . $vehicule->getModele() . ' / ' . $vehicule->getImmatriculation();
                },
                'query_builder' => function (EntityRepository $er) {

                    $societe = $this->entityManager->getRepository("App\Entity\UserSociete")->findOneBy(array('user' => $this->getUser()));

                    return $er->createQueryBuilder('s')
                        ->select('v')
                        ->from('App\Entity\Vehicule', 'v')
                        ->andWhere('v.societe= :societe')
                        ->andWhere('v.statut= :statut')
                        ->setParameter('statut', 1)
                        ->setParameter('societe', $societe->getSociete());

                }
            ])
            ->add('villedepart', EntityType::class, [
                'class' => Ville::class,
                'choice_label' => 'nom',
                'query_builder' => function (EntityRepository $er) {

                    $societe = $this->entityManager->getRepository("App\Entity\UserSociete")->findOneBy(array('user' => $this->getUser()));

                    return $er->createQueryBuilder('s')
                        ->select('v')
                        ->from('App\Entity\Ville', 'v')
                        ->andWhere('v.societe= :societe')
                        ->setParameter('societe', $societe->getSociete());

                }
            ])
            ->add('villearrive', EntityType::class, [
                'class' => Ville::class,
                'choice_label' => 'nom',
                'query_builder' => function (EntityRepository $er) {

                    $societe = $this->entityManager->getRepository("App\Entity\UserSociete")->findOneBy(array('user' => $this->getUser()));


                    return $er->createQueryBuilder('s')
                        ->select('v')
                        ->from('App\Entity\Ville', 'v')
                        ->andWhere('v.societe= :societe')
                        ->setParameter('societe', $societe->getSociete());

                }
            ])
            ->add('chauffeur_temp', EntityType::class, [
                'class' => Chauffeur::class,
                'choice_label' => fn(Chauffeur $c) => $c->getPrenom() . ' ' . $c->getNom(),
                'mapped' => false,
                'required' => false,
                'label' => 'Sélectionner un chauffeur',
            ])
            ->add('chauffeurs_ids', HiddenType::class, [
                'mapped' => false,
            ]);

        $form->handleRequest($request);

        $forms = $this->createForm(TransitType::class, $transit);
        $forms->remove('numeroetape');
        $forms->remove('datearrive');
        $forms->remove('datedepart');
        $forms->remove('heurearrive');
        $forms->remove('heuredepart');

        $forms->handleRequest($request);


        $today = new \DateTime();

        // $items = $today->format("Y-m-d");


        if ($form->isSubmitted() && $form->isValid()) {

            $idsString = $form->get('chauffeurs_ids')->getData();
            $ids = array_filter(explode(',', $idsString));

            foreach ($ids as $chauffeurId) {
                $chauffeur = $entityManager->getRepository(Chauffeur::class)->find($chauffeurId);
                if ($chauffeur) {
                    $cv = new ChauffeurVoyage();
                    $cv->setChauffeur($chauffeur);
                    $cv->setVoyage($voyage);
                    $cv->setStatut(true);
                    $voyage->addChauffeurVoyage($cv);
                }
            }

            $voyage->setDatecreation($today);
            $voyage->setStatut(1);
            $voyage->setActif(true);
            $voyage->setSociete($societe->getSociete());
            $voyage->setAvailable($voyage->getVehicule()->getCapacite());
            $entityManager->persist($voyage);


            $transit->setStatut(1);
            $transit->setVoyage($voyage);
            $transit->setVille($voyage->getVilledepart());
            $transit->setVillearrive($voyage->getVillearrive());
            $transit->setNumeroetape(1);
            $transit->setDatedepart($voyage->getDatedepart());
            $transit->setDatearrive($voyage->getDatearrive());
            $transit->setHeuredepart($voyage->getHeuredepart());
            $transit->setHeurearrive($voyage->getHeurearrivee());
            $transit->setDatecreation($today);
            $entityManager->persist($transit);


            $entityManager->flush();


            $entityManager->flush();

            return $this->redirectToRoute('app_voyage_index', [], Response::HTTP_SEE_OTHER);
        }

        $response = new Response(null, $form->isSubmitted() ? 422 : 200);


        return $this->render('voyage/new.html.twig', [
            'voyage' => $voyage,
            'form' => $form,
            'forms' => $forms,
            'transit' => $transit->getId(),
        ], $response);
    }

    #[Route('/{id}', name: 'app_voyage_show', methods: ['GET'])]
    public function show(Voyage $voyage): Response
    {
        $canView = $this->isGranted('VOYAGE_VIEW');          // ou $this->isGranted('VOYAGE_VIEW', $subject)
        $canCreate = $this->isGranted('VOYAGE_CREATE');          // ou $this->isGranted('VOYAGE_LIST', $subject)
        $canUpdate = $this->isGranted('VOYAGE_UPDATE');
        $canDelete = $this->isGranted('VOYAGE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canView && !$canCreate && !$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        return $this->render('voyage/show.html.twig', [
            'voyage' => $voyage,
        ]);
    }

    #[Route('/{id}/edit', name: 'app_voyage_edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Voyage $voyage, EntityManagerInterface $entityManager): Response
    {

        $canUpdate = $this->isGranted('VOYAGE_UPDATE');
        $canDelete = $this->isGranted('VOYAGE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');

        if (!$canUpdate && !$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $form = $this->createForm(VoyageType::class, $voyage);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->flush();

            return $this->redirectToRoute('app_voyage_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('voyage/edit.html.twig', [
            'voyage' => $voyage,
            'form' => $form,
        ]);
    }


    #[Route('/actif/{id}/actif/actif/1/1/1', name: 'app_voyage_edit_desactiver', methods: ['GET', 'POST'])]
    public function editDesactiver(Request $request, Voyage $voyage, EntityManagerInterface $entityManager): Response
    {


        $canUpdate = $this->isGranted('VOYAGE_UPDATE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canUpdate && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        $voyage->setActif(false);
        $entityManager->flush();

        return $this->redirectToRoute('app_voyage_index', [], Response::HTTP_SEE_OTHER);
    }


    #[Route('/{id}', name: 'app_voyage_delete', methods: ['POST'])]
    public function delete(Request $request, Voyage $voyage, EntityManagerInterface $entityManager): Response
    {


        $canDelete = $this->isGranted('VOYAGE_DELETE');
        $superAmin = $this->isGranted('SUPER_ADMIN');


        if (!$canDelete && !$superAmin) {
            throw $this->createAccessDeniedException();
        }

        if ($this->isCsrfTokenValid('delete' . $voyage->getId(), $request->getPayload()->get('_token'))) {
            $entityManager->remove($voyage);
            $entityManager->flush();
        }

        return $this->redirectToRoute('app_voyage_index', [], Response::HTTP_SEE_OTHER);
    }
}
